// E-wallet Account Inquiry Service - Real Account Check
const axios = require('axios');

class EwalletService {
    constructor() {
        this.isEnabled = true;
        this.apiBaseUrl = 'https://cekrekening-api.belibayar.online/api/v1/account-inquiry';
        this.requestTimeout = 15000; // 15 detik timeout
          // Mapping channel IDs untuk berbagai e-wallet/bank sesuai dokumentasi API
        this.channelIds = {
            // E-wallets
            'dana': 'dana',
            'ovo': 'ovo', 
            'gopay': 'gopay',
            'shopeepay': 'shopeepay',
            'linkaja': 'linkaja',
            
            // Banks - menggunakan value dari dokumentasi
            'bca': 'bca',
            'mandiri': 'mandiri',
            'bni': 'bni',
            'bri': 'bri',
            'cimb': 'cimb',
            'permata': 'permata',
            'danamon': 'danamon',
            'mega': 'mega',
            'panin': 'panin',
            'ocbc': 'ocbc',
            'uob': 'uob',
            'hsbc': 'hsbc',
            'standard_chartered': 'standard_chartered',
            'commonwealth': 'commonwealth',
            'citibank': 'citibank',
            'dbs': 'dbs',
            'btn': 'btn',
            'bukopin': 'bukopin',
            'maybank': 'bii',
            'bjb': 'bjb'
        };
          // Daftar channel yang didukung
        this.supportedChannels = Object.keys(this.channelIds);
    }
    
    // Extract phone number and channel from message
    extractAccountInfo(message) {
        const accountRegex = /(\+?62|0)?(\d{10,15})/g;
        const matches = message.match(accountRegex);
        
        // Cari kata kunci channel/bank
        const messageWords = message.toLowerCase().split(/\s+/);
        let detectedChannel = null;
        
        for (const word of messageWords) {
            if (this.supportedChannels.includes(word)) {
                detectedChannel = word;
                break;
            }
            // Cek alternatif nama
            if (word.includes('mandiri')) detectedChannel = 'mandiri';
            else if (word.includes('dana')) detectedChannel = 'dana';
            else if (word.includes('ovo')) detectedChannel = 'ovo';
            else if (word.includes('gopay')) detectedChannel = 'gopay';
            else if (word.includes('shopee')) detectedChannel = 'shopeepay';
            else if (word.includes('linkaja') || word.includes('link aja')) detectedChannel = 'linkaja';
        }
          return {
            accountNumber: matches ? matches[0].replace(/\D/g, '') : null,
            channel: detectedChannel
        };
    }
      // Panggil API untuk cek nama pemilik rekening
    async checkAccountOwner(accountNumber, channel) {
        try {
            if (!accountNumber || !channel) {
                throw new Error('Nomor rekening dan channel harus disediakan');
            }
            
            // Gunakan channel langsung sesuai dokumentasi API
            const accountBank = this.channelIds[channel] || channel;
            
            console.log(`[E-wallet API] Checking account: ${accountNumber} on ${channel} (Bank Code: ${accountBank})`);
            
            const response = await axios.post(this.apiBaseUrl, {
                account_number: accountNumber,
                account_bank: accountBank
            }, {
                timeout: this.requestTimeout,
                headers: {
                    'Content-Type': 'application/json',
                    'User-Agent': 'WhatsApp-Bot/1.0'
                }
            });
              console.log(`[E-wallet API] Response status: ${response.status}`);
            console.log(`[E-wallet API] Response data:`, response.data);
            
            // Cek jika response success dan ada data sesuai dokumentasi
            if (response.data && response.data.success && response.data.data && response.data.data.account_holder) {
                return {
                    success: true,
                    accountNumber: accountNumber,
                    channel: channel,
                    ownerName: response.data.data.account_holder,
                    status: 'active'
                };
            } 
            // Jika success = false atau data kosong
            else if (response.data && !response.data.success) {
                return {
                    success: false,
                    accountNumber: accountNumber, 
                    channel: channel,
                    error: response.data.message || 'Account not found'
                };
            }
            // Response error lainnya
            else {
                return {
                    success: false,
                    accountNumber: accountNumber, 
                    channel: channel,
                    error: response.data?.message || 'Rekening tidak ditemukan'
                };
            }
              } catch (error) {
            console.error('[E-wallet API] Error:', error.message);
            
            // Handle 404 response (account not found)
            if (error.response && error.response.status === 404) {
                const responseData = error.response.data;
                return {
                    success: false,
                    accountNumber: accountNumber,
                    channel: channel,
                    error: responseData?.message || 'Rekening tidak ditemukan'
                };
            }
            
            // Handle timeout
            if (error.code === 'ECONNABORTED') {
                return {
                    success: false,
                    accountNumber: accountNumber,
                    channel: channel,
                    error: 'Timeout - API tidak merespons'
                };
            }
            
            // Handle network error
            if (error.code === 'ENOTFOUND' || error.code === 'ECONNREFUSED') {
                return {
                    success: false,
                    accountNumber: accountNumber,
                    channel: channel,
                    error: 'Tidak dapat terhubung ke server - coba lagi nanti'
                };
            }
            
            return {
                success: false,
                accountNumber: accountNumber,
                channel: channel,
                error: error.response?.data?.message || error.message || 'Gagal mengecek rekening'
            };
        }
    }    // Format hasil pengecekan rekening
    formatAccountResult(result) {
        if (result.success) {
            return `✅ **Hasil Cek Rekening:**

💳 **Nomor Rekening:** ${result.accountNumber}
🏦 **Platform:** ${result.channel.toUpperCase()}
👤 **Nama Pemilik:** ${result.ownerName}
📊 **Status:** Aktif

${this.getSafetyTips()}`;
        } else {            return `❌ **Gagal Cek Rekening:**

💳 **Nomor:** ${result.accountNumber}
🏦 **Platform:** ${result.channel ? result.channel.toUpperCase() : 'Unknown'}
⚠️ **Error:** ${result.error}

💡 **Kemungkinan Penyebab:**
• Nomor rekening tidak valid atau salah
• Rekening tidak aktif atau sudah ditutup
• Platform/bank yang dipilih tidak sesuai
• Rekening baru yang belum terdaftar di sistem

🔍 **Tips Pengecekan:**
• Pastikan nomor rekening lengkap dan benar
• Coba platform/bank yang berbeda jika tidak yakin
• Minta konfirmasi nomor rekening dari pemilik

${this.getSafetyTips()}`;
        }
    }

    // Daftar platform yang didukung
    getSupportedPlatforms() {
        const platforms = {
            'E-wallet': ['DANA', 'OVO', 'GoPay', 'ShopeePay', 'LinkAja'],
            'Bank': ['BCA', 'Mandiri', 'BNI', 'BRI', 'Jenius', 'CIMB', 'Permata', 'Danamon', 'Mega', 'Maybank', 'OCBC', 'Panin', 'UOB', 'HSBC', 'Standard Chartered', 'Commonwealth', 'Citibank', 'DBS']
        };
        
        let result = `💳 **Platform yang Didukung:**\n\n`;
        
        for (const [category, items] of Object.entries(platforms)) {
            result += `**${category}:**\n`;
            result += items.map(item => `• ${item}`).join('\n') + '\n\n';
        }
        
        result += `📝 **Cara Pakai:**\n`;
        result += `• "cek dana 081234567890"\n`;
        result += `• "cek bca 1234567890"\n`;
        result += `• "verifikasi ovo 081234567890"\n`;
        
        return result;
    }    // Safety tips untuk transaksi aman
    getSafetyTips() {
        return `🛡️ **Tips Transaksi Aman:**

✅ **Setelah Cek Nama:**
• Pastikan nama sesuai dengan identitas seller
• Cross-check dengan akun media sosial
• Gunakan platform jual-beli terpercaya
• Minta bukti identitas tambahan jika perlu

❌ **Waspada Penipuan:**
• Nama rekening tidak sesuai dengan seller
• Minta transfer duluan tanpa jaminan
• Menolak COD atau sistem escrow
• Harga terlalu murah dari pasaran

🔒 **Best Practice:**
• Gunakan rekber untuk transaksi besar
• Simpan screenshot percakapan
• Cek review/testimoni seller
• Jangan transfer ke rekening atas nama orang lain`;
    }

    // Cek apakah pesan meminta pengecekan rekening
    shouldProcessEwalletCheck(message) {
        const keywords = ['cek', 'check', 'verifikasi', 'dana', 'ovo', 'gopay', 'bca', 'mandiri', 'bni', 'bri', 'shopeepay', 'linkaja'];
        const accountPattern = /\d{10,15}/;
        
        const messageWords = message.toLowerCase().split(/\s+/);
        const hasKeyword = keywords.some(keyword => 
            messageWords.some(word => word.includes(keyword))
        );
        
        return hasKeyword && accountPattern.test(message);
    }

    // Main method untuk handle pengecekan rekening
    async handleAccountCheck(message) {
        try {
            const accountInfo = this.extractAccountInfo(message);
            
            if (!accountInfo.accountNumber) {
                return `❌ **Nomor rekening tidak ditemukan**

📝 **Format yang benar:**
• "cek dana 081234567890"
• "verifikasi bca 1234567890"
• "check ovo 081234567890"

${this.getSupportedPlatforms()}`;
            }
            
            if (!accountInfo.channel) {
                return `❌ **Platform tidak dikenali**

💡 Sebutkan platform/bank yang ingin dicek:

${this.getSupportedPlatforms()}`;
            }
            
            console.log(`[E-wallet] Processing check: ${accountInfo.accountNumber} on ${accountInfo.channel}`);
            
            // Kirim loading message
            const loadingMessage = `🔍 **Mengecek rekening...**\n\n💳 Nomor: ${accountInfo.accountNumber}\n🏦 Platform: ${accountInfo.channel.toUpperCase()}\n\n⏳ Mohon tunggu...`;
            
            // Call API
            const result = await this.checkAccountOwner(accountInfo.accountNumber, accountInfo.channel);
            
            return this.formatAccountResult(result);
            
        } catch (error) {
            console.error('[E-wallet] Error handling account check:', error);
            return `❌ **Terjadi kesalahan sistem**

Silakan coba lagi atau hubungi admin jika masalah berlanjut.

${this.getSupportedPlatforms()}`;
        }
    }
}

module.exports = EwalletService;
