// Health Check Server untuk Production Monitoring
const express = require('express');
const QRCode = require('qrcode');
const fs = require('fs');
const path = require('path');

class HealthCheckServer {
    constructor() {
        this.app = express();
        this.currentQR = null;
        this.botStatus = 'starting';
        this.botStartTime = Date.now();
        this.botProcess = null;
        this.setupMiddleware();
        this.setupRoutes();
    }

    setupMiddleware() {
        // Enable JSON parsing with error handling
        this.app.use(express.json({ 
            limit: '10mb',
            type: 'application/json'
        }));
        this.app.use(express.urlencoded({ 
            extended: true,
            limit: '10mb'
        }));
        
        // Add JSON error handler
        this.app.use((error, req, res, next) => {
            if (error instanceof SyntaxError && error.status === 400 && 'body' in error) {
                console.error('Bad JSON:', error.message);
                return res.status(400).json({ error: 'Invalid JSON format' });
            }
            next();
        });
        
        // Serve static files (limited to specific paths to avoid conflicts)
        this.app.use('/assets', express.static('.'));
    }

    setupRoutes() {
        // Main configuration page
        this.app.get('/', (req, res) => {
            res.sendFile(path.join(__dirname, 'config.html'));
        });

        // Configuration API endpoints
        this.app.get('/api/config', (req, res) => {
            try {
                res.setHeader('Content-Type', 'application/json');
                const config = this.getCurrentConfig();
                res.json(config);
            } catch (error) {
                console.error('Config API error:', error);
                res.status(500).json({ error: error.message });
            }
        });

        this.app.post('/api/config', (req, res) => {
            try {
                res.setHeader('Content-Type', 'application/json');
                this.saveConfig(req.body);
                res.json({ message: 'Konfigurasi berhasil disimpan', success: true });
                
                // Restart bot with new config
                setTimeout(() => {
                    this.restartBot();
                }, 1000);
            } catch (error) {
                console.error('Save config error:', error);
                res.status(500).json({ error: error.message });
            }
        });

        // Bot control API
        this.app.post('/api/bot/start', (req, res) => {
            try {
                res.setHeader('Content-Type', 'application/json');
                this.startBot();
                res.json({ message: '🚀 Bot berhasil dimulai', success: true });
            } catch (error) {
                console.error('Start bot error:', error);
                res.status(500).json({ error: error.message });
            }
        });

        this.app.post('/api/bot/restart', (req, res) => {
            try {
                res.setHeader('Content-Type', 'application/json');
                this.restartBot();
                res.json({ message: '🔄 Bot berhasil direstart', success: true });
            } catch (error) {
                console.error('Restart bot error:', error);
                res.status(500).json({ error: error.message });
            }
        });

        this.app.post('/api/bot/stop', (req, res) => {
            try {
                res.setHeader('Content-Type', 'application/json');
                this.stopBot();
                res.json({ message: '⏹️ Bot berhasil dihentikan', success: true });
            } catch (error) {
                console.error('Stop bot error:', error);
                res.status(500).json({ error: error.message });
            }
        });

        // Status API
        this.app.get('/api/status', (req, res) => {
            try {
                res.setHeader('Content-Type', 'application/json');
                const status = {
                    bot_running: this.botStatus === 'connected' || this.botStatus === 'ready',
                    bot_status: this.botStatus,
                    uptime_seconds: Math.floor((Date.now() - this.botStartTime) / 1000),
                    connected_to_whatsapp: global.whatsappSocket?.user?.id ? true : false,
                    memory_usage_mb: Math.round(process.memoryUsage().heapUsed / 1024 / 1024),
                    environment: process.env.NODE_ENV || 'development',
                    last_restart: new Date(this.botStartTime).toLocaleString('id-ID')
                };
                res.json(status);
            } catch (error) {
                console.error('Status API error:', error);
                res.status(500).json({ error: error.message });
            }
        });

        // Health check endpoint untuk monitoring
        this.app.get('/health', (req, res) => {
            try {
                res.setHeader('Content-Type', 'application/json');
                res.json({
                    status: 'OK',
                    bot_status: this.botStatus,
                    uptime: Math.floor((Date.now() - this.botStartTime) / 1000),
                    timestamp: new Date().toISOString(),
                    memory_usage: process.memoryUsage(),
                    connected: global.whatsappSocket?.user?.id ? true : false,
                    environment: process.env.NODE_ENV || 'development'
                });
            } catch (error) {
                console.error('Health check error:', error);
                res.status(500).json({ 
                    status: 'ERROR',
                    error: error.message,
                    timestamp: new Date().toISOString()
                });
            }
        });

        // QR Code web interface
        this.app.get('/qr', async (req, res) => {
            if (!this.currentQR) {
                return res.send(`
                    <html>
                        <head>
                            <title>WhatsApp Bot QR</title>
                            <meta charset="UTF-8">
                            <meta name="viewport" content="width=device-width, initial-scale=1.0">
                            <style>
                                body { font-family: Arial, sans-serif; text-align: center; margin: 50px; }
                                .container { max-width: 400px; margin: 0 auto; }
                                .status { color: #28a745; font-weight: bold; }
                            </style>
                        </head>
                        <body>
                            <div class="container">
                                <h2>🤖 Mas Angga WhatsApp Bot</h2>
                                <p class="status">✅ Bot sudah terhubung!</p>
                                <p>Tidak perlu scan QR code lagi.</p>
                                <p><a href="/health">Cek Status Bot</a></p>
                            </div>
                        </body>
                    </html>
                `);
            }
            
            try {
                const qrImage = await QRCode.toDataURL(this.currentQR);
                res.send(`
                    <html>
                        <head>
                            <title>WhatsApp Bot QR</title>
                            <meta charset="UTF-8">
                            <meta name="viewport" content="width=device-width, initial-scale=1.0">
                            <meta http-equiv="refresh" content="30">
                            <style>
                                body { font-family: Arial, sans-serif; text-align: center; margin: 20px; }
                                .container { max-width: 400px; margin: 0 auto; }
                                img { max-width: 300px; border: 2px solid #ddd; border-radius: 10px; }
                                .instructions { margin: 20px 0; color: #666; }
                            </style>
                        </head>
                        <body>
                            <div class="container">
                                <h2>🤖 Mas Angga Bot - QR Code</h2>
                                <img src="${qrImage}" alt="WhatsApp QR Code">
                                <div class="instructions">
                                    <p>📱 <strong>Cara Scan:</strong></p>
                                    <p>1. Buka WhatsApp di HP</p>
                                    <p>2. Tekan Menu (3 titik) → Linked devices</p>
                                    <p>3. Tekan "Link a device"</p>
                                    <p>4. Scan QR code di atas</p>
                                </div>
                                <p><small>🔄 Halaman refresh otomatis setiap 30 detik</small></p>
                            </div>
                        </body>
                    </html>
                `);
            } catch (error) {
                console.error('Error generating QR code:', error);
                res.status(500).send(`
                    <html>
                        <head><title>Error</title></head>
                        <body style="text-align:center; font-family:Arial;">
                            <h2>❌ Error</h2>
                            <p>Gagal generate QR code. Silakan refresh halaman.</p>
                        </body>
                    </html>
                `);
            }
        });

        // Dashboard endpoint
        this.app.get('/dashboard', (req, res) => {
            res.send(`
                <html>
                    <head>
                        <title>Mas Angga WhatsApp Bot</title>
                        <meta charset="UTF-8">
                        <meta name="viewport" content="width=device-width, initial-scale=1.0">
                        <style>
                            body { font-family: Arial, sans-serif; text-align: center; margin: 50px; background: #f5f5f5; }
                            .container { max-width: 500px; margin: 0 auto; background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
                            h1 { color: #25D366; }
                            .link { display: inline-block; margin: 10px; padding: 10px 20px; background: #25D366; color: white; text-decoration: none; border-radius: 5px; }
                            .link:hover { background: #1da851; }
                        </style>
                    </head>
                    <body>
                        <div class="container">
                            <h1>🤖 Mas Angga WhatsApp Bot</h1>
                            <p>Bot WhatsApp AI dengan fitur lengkap:</p>
                            <ul style="text-align: left;">
                                <li>💬 Chat AI (Curhat & Edukasi)</li>
                                <li>🎥 Download TikTok tanpa watermark</li>
                                <li>💳 Verifikasi rekening e-wallet/bank</li>
                            </ul>
                            <div>
                                <a href="/qr" class="link">📱 Scan QR Code</a>
                                <a href="/health" class="link">🔍 Check Status</a>
                                <a href="/" class="link">⚙️ Konfigurasi</a>
                            </div>
                            <p><small>Status: ${this.botStatus}</small></p>
                        </div>
                    </body>
                </html>
            `);
        });
    }

    // Configuration management
    getCurrentConfig() {
        const config = {
            bot_name: process.env.BOT_NAME || 'Mas Angga',
            port: process.env.PORT || '3000',
            memory_limit: process.env.MEMORY_LIMIT || '512',
            groq_api_key: process.env.GROQ_API_KEY || '',
            node_env: process.env.NODE_ENV || 'production'
        };
        
        return config;
    }

    saveConfig(newConfig) {
        const envPath = path.join(__dirname, '.env');
        let envContent = '';

        // Required config
        envContent += `NODE_ENV=production\n`;
        envContent += `BOT_NAME=${newConfig.bot_name || 'Mas Angga'}\n`;
        envContent += `PORT=${newConfig.port || '3000'}\n`;
        envContent += `GROQ_API_KEY=${newConfig.groq_api_key || ''}\n`;
        envContent += `MAX_RETRIES=3\n`;
        envContent += `REQUEST_TIMEOUT=30000\n`;
        envContent += `MEMORY_LIMIT=${newConfig.memory_limit || '512'}\n`;

        fs.writeFileSync(envPath, envContent);
        
        // Update process.env
        process.env.BOT_NAME = newConfig.bot_name || 'Mas Angga';
        process.env.PORT = newConfig.port || '3000';
        process.env.GROQ_API_KEY = newConfig.groq_api_key || '';
        process.env.MEMORY_LIMIT = newConfig.memory_limit || '512';
    }

    // Bot control methods
    startBot() {
        if (this.botProcess) {
            throw new Error('Bot sudah berjalan');
        }

        const { spawn } = require('child_process');
        this.botProcess = spawn('node', ['index-production.js'], {
            stdio: 'inherit',
            cwd: __dirname
        });

        this.botProcess.on('close', (code) => {
            console.log(`Bot process exited with code ${code}`);
            this.botProcess = null;
            this.botStatus = 'stopped';
        });

        this.botStatus = 'starting';
        this.botStartTime = Date.now();
    }

    stopBot() {
        if (this.botProcess) {
            this.botProcess.kill('SIGTERM');
            this.botProcess = null;
            this.botStatus = 'stopped';
        }
    }

    restartBot() {
        this.stopBot();
        setTimeout(() => {
            this.startBot();
        }, 2000);
    }

    updateQR(qrCode) {
        this.currentQR = qrCode;
    }

    updateStatus(status) {
        this.botStatus = status;
    }

    start(port = 3000) {
        this.app.listen(port, () => {
            console.log(`🔍 Health check server running on port ${port}`);
            console.log(`📱 QR Code web interface: http://localhost:${port}/qr`);
            console.log(`🏠 Bot dashboard: http://localhost:${port}/dashboard`);
            console.log(`⚙️ Configuration page: http://localhost:${port}/`);
        });
    }
}

module.exports = HealthCheckServer;
