const axios = require('axios');

class TikTokService {
    constructor() {
        this.isEnabled = true;
    }

    // Check if URL is a valid TikTok URL
    isTikTokUrl(url) {
        const tiktokRegex = /(?:https?:\/\/)?(?:www\.)?(vm\.|vt\.|m\.)?tiktok\.com\/[@A-Za-z0-9._-]+\/?(.*)?/;
        const tiktokShortRegex = /(?:https?:\/\/)?(vm\.|vt\.)tiktok\.com\/[A-Za-z0-9]+/;
        return tiktokRegex.test(url) || tiktokShortRegex.test(url);
    }

    // Extract TikTok URL from message
    extractTikTokUrl(message) {
        const urlRegex = /(https?:\/\/[^\s]+)/g;
        const urls = message.match(urlRegex);
        
        if (urls) {
            return urls.find(url => this.isTikTokUrl(url));
        }
        return null;
    }

    // Download TikTok video using free API
    async downloadWithAPI(url) {
        try {
            console.log('🎬 Downloading TikTok video with free API...');
            
            // Clean the URL first
            const cleanUrl = url.replace(/\?.*$/, ''); // Remove query parameters
            
            // Try tikwm.com API
            const apiUrl = `https://tikwm.com/api/?url=${encodeURIComponent(cleanUrl)}`;
            
            const response = await axios.get(apiUrl, {
                timeout: 15000,
                headers: {
                    'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'
                }
            });
            
            if (response.data && response.data.code === 0 && response.data.data) {
                const data = response.data.data;
                return {
                    success: true,
                    data: {
                        title: data.title || 'TikTok Video',
                        author: data.author?.nickname || data.author?.unique_id || 'Unknown',
                        duration: data.duration || 'Unknown',
                        videoUrl: data.play || data.wmplay || data.hdplay,
                        audioUrl: data.music || null,
                        thumbnail: data.cover || data.origin_cover || null,
                        stats: {
                            views: data.play_count || 0,
                            likes: data.digg_count || 0,
                            comments: data.comment_count || 0,
                            shares: data.share_count || 0
                        }
                    }
                };
            }
            
            throw new Error('No video data found from API');
        } catch (error) {
            console.error('❌ API downloader failed:', error.message);
            throw error;
        }
    }

    // Fallback method with basic info
    async downloadWithFallback(url) {
        try {
            console.log('🎬 Using fallback method...');
            
            return {
                success: true,
                data: {
                    title: 'TikTok Video',
                    author: 'TikTok User',
                    duration: 'Unknown',
                    videoUrl: null, // No direct video URL
                    audioUrl: null,
                    thumbnail: null,
                    stats: {
                        views: 0,
                        likes: 0,
                        comments: 0,
                        shares: 0
                    }
                }
            };
        } catch (error) {
            console.error('❌ Fallback failed:', error.message);
            throw error;
        }
    }

    // Main download function with fallback
    async downloadVideo(url) {
        if (!url) {
            throw new Error('URL tidak valid');
        }

        if (!this.isTikTokUrl(url)) {
            throw new Error('URL bukan dari TikTok');
        }

        console.log('📱 Memproses URL TikTok:', url);

        // Try API first
        try {
            return await this.downloadWithAPI(url);
        } catch (error1) {
            console.log('⚠️ API failed, using fallback...');
            
            // Try fallback method
            try {
                return await this.downloadWithFallback(url);
            } catch (error2) {
                console.error('❌ All methods failed:', {
                    api: error1.message,
                    fallback: error2.message
                });
                
                throw new Error('Gagal mengunduh video TikTok. Silakan coba lagi nanti.');
            }
        }
    }

    // Format video info for WhatsApp message
    formatVideoInfo(data) {
        const stats = data.stats;
        const formatNumber = (num) => {
            if (num >= 1000000) return (num / 1000000).toFixed(1) + 'M';
            if (num >= 1000) return (num / 1000).toFixed(1) + 'K';
            return num.toString();
        };

        return `🎬 *TikTok Video*

📝 *Judul:* ${data.title}
👤 *Author:* ${data.author}
⏱️ *Durasi:* ${data.duration}

📊 *Statistik:*
👀 Views: ${formatNumber(stats.views)}
❤️ Likes: ${formatNumber(stats.likes)}
💬 Comments: ${formatNumber(stats.comments)}
📤 Shares: ${formatNumber(stats.shares)}

✅ Info video berhasil diambil!`;
    }

    // Check if message contains TikTok URL and should be processed
    shouldProcessMessage(message) {
        return this.extractTikTokUrl(message) !== null;
    }
}

module.exports = TikTokService;
